/*
 * Copyright 2010-2025 Hyland Software, Inc. and its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.activiti.engine.impl.cmd;

import java.io.Serializable;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import org.activiti.bpmn.model.ValuedDataObject;
import org.activiti.engine.impl.ProcessInstanceCreationOptions;
import org.activiti.engine.impl.interceptor.Command;
import org.activiti.engine.impl.interceptor.CommandContext;
import org.activiti.engine.impl.persistence.deploy.DeploymentManager;
import org.activiti.engine.impl.runtime.ProcessInstanceBuilderImpl;
import org.activiti.engine.impl.util.ProcessDefinitionRetriever;
import org.activiti.engine.impl.util.ProcessInstanceHelper;
import org.activiti.engine.repository.ProcessDefinition;
import org.activiti.engine.runtime.ProcessInstance;

/**


 */
public class StartProcessInstanceCmd<T> implements Command<ProcessInstance>, Serializable {

    private static final long serialVersionUID = 1L;
    protected String processDefinitionKey;
    protected String processDefinitionId;
    protected Map<String, Object> variables;
    protected Map<String, Object> transientVariables;
    protected String businessKey;
    protected String tenantId;
    protected String processInstanceName;
    protected ProcessInstanceHelper processInstanceHelper;
    protected String linkedProcessInstanceId;
    protected String linkedProcessInstanceType;

    public StartProcessInstanceCmd(
        String processDefinitionKey,
        String processDefinitionId,
        String businessKey,
        Map<String, Object> variables
    ) {
        this.processDefinitionKey = processDefinitionKey;
        this.processDefinitionId = processDefinitionId;
        this.businessKey = businessKey;
        this.variables = variables;
    }

    public StartProcessInstanceCmd(
        String processDefinitionKey,
        String processDefinitionId,
        String businessKey,
        Map<String, Object> variables,
        String tenantId
    ) {
        this(processDefinitionKey, processDefinitionId, businessKey, variables);
        this.tenantId = tenantId;
    }

    public StartProcessInstanceCmd(ProcessInstanceBuilderImpl processInstanceBuilder) {
        this(
            processInstanceBuilder.getProcessDefinitionKey(),
            processInstanceBuilder.getProcessDefinitionId(),
            processInstanceBuilder.getBusinessKey(),
            processInstanceBuilder.getVariables(),
            processInstanceBuilder.getTenantId()
        );
        this.processInstanceName = processInstanceBuilder.getProcessInstanceName();
        this.transientVariables = processInstanceBuilder.getTransientVariables();
        this.linkedProcessInstanceId = processInstanceBuilder.getLinkedProcessInstanceId();
        this.linkedProcessInstanceType = processInstanceBuilder.getLinkedProcessInstanceType();
  }

    public ProcessInstance execute(CommandContext commandContext) {
        DeploymentManager deploymentCache = commandContext.getProcessEngineConfiguration().getDeploymentManager();

        ProcessDefinitionRetriever processRetriever = new ProcessDefinitionRetriever(this.tenantId, deploymentCache);
        ProcessDefinition processDefinition = processRetriever.getProcessDefinition(
            this.processDefinitionId,
            this.processDefinitionKey
        );

        processInstanceHelper = commandContext.getProcessEngineConfiguration().getProcessInstanceHelper();

        ProcessInstanceCreationOptions options = ProcessInstanceCreationOptions
            .builder(processDefinition)
            .businessKey(businessKey)
            .processInstanceName(processInstanceName)
            .variables(variables)
            .transientVariables(transientVariables)
            .linkedProcessInstanceId(linkedProcessInstanceId)
            .linkedProcessInstanceType(linkedProcessInstanceType)
            .build();

        return createAndStartProcessInstance(options);
    }

    protected ProcessInstance createAndStartProcessInstance(
        ProcessInstanceCreationOptions options
    ) {
        return processInstanceHelper.createAndStartProcessInstance(options);
    }

    protected Map<String, Object> processDataObjects(Collection<ValuedDataObject> dataObjects) {
        Map<String, Object> variablesMap = new HashMap<String, Object>();
        // convert data objects to process variables
        if (dataObjects != null) {
            for (ValuedDataObject dataObject : dataObjects) {
                variablesMap.put(dataObject.getName(), dataObject.getValue());
            }
        }
        return variablesMap;
    }
}
