/*
 * Decompiled with CFR 0.152.
 */
package org.alfresco.transform.misc.transformers;

import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiFunction;
import java.util.function.BiPredicate;
import java.util.function.Function;
import java.util.stream.Stream;
import javax.imageio.ImageIO;
import javax.imageio.ImageReader;
import javax.imageio.stream.ImageInputStream;
import org.alfresco.transform.base.TransformManager;
import org.alfresco.transform.misc.transformers.ImageToPdfTransformer;
import org.alfresco.transform.misc.util.ArgumentsCartesianProduct;
import org.apache.pdfbox.Loader;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.pdmodel.common.PDRectangle;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;
import org.mockito.BDDMockito;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

class ImageToPdfTransformerTest {
    private static final File sourceFile = ImageToPdfTransformerTest.loadFile("sample.gif");
    private static final File SOURCE_TIFF_FILE = ImageToPdfTransformerTest.loadFile("sample.tiff");
    private static final int sourceFileWidth;
    private static final int sourceFileHeight;
    @Mock
    private TransformManager transformManager;
    @InjectMocks
    private ImageToPdfTransformer transformer;
    private File targetFile = null;

    ImageToPdfTransformerTest() {
    }

    @BeforeEach
    void setUp() throws IOException {
        MockitoAnnotations.openMocks((Object)this);
        this.targetFile = File.createTempFile("temp_target", ".pdf");
    }

    static Stream<ImageFile> imageFiles() {
        return Stream.of(ImageFile.of("sample.jpg", "image/jpeg"), ImageFile.of("sample.gif", "image/gif"), ImageFile.of("sample.png", "image/png"));
    }

    static Stream<TransformOptions> defaultTransformOptions() {
        return Stream.of(TransformOptions.none(), TransformOptions.of(0, null), TransformOptions.of(0, 0));
    }

    static Stream<TransformOptions> tiffTransformOptions() {
        return Stream.of(TransformOptions.of(0, 0), TransformOptions.of(0, 1), TransformOptions.of(1, 1), TransformOptions.of(null, 0), TransformOptions.of(null, 1), TransformOptions.of(0, null), TransformOptions.of(1, null), TransformOptions.none());
    }

    static Stream<Arguments> transformSourcesAndOptions() {
        ImageFile tiffImage = ImageFile.of("sample.tiff", "image/tiff", 6);
        return Stream.of(ArgumentsCartesianProduct.of(ImageToPdfTransformerTest.imageFiles(), ImageToPdfTransformerTest.defaultTransformOptions()), ArgumentsCartesianProduct.of(tiffImage, ImageToPdfTransformerTest.tiffTransformOptions())).flatMap(Function.identity());
    }

    @ParameterizedTest
    @MethodSource(value={"transformSourcesAndOptions"})
    void testTransformImageToPdf(ImageFile imageFile, TransformOptions transformOptions) throws Exception {
        File sourceFile = ImageToPdfTransformerTest.loadFile(imageFile.fileName);
        this.transformer.transform(imageFile.mimetype, "application/pdf", transformOptions.toMap(), sourceFile, this.targetFile, this.transformManager);
        BDDMockito.then((Object)this.transformManager).shouldHaveNoInteractions();
        try (PDDocument actualPdfDocument = Loader.loadPDF((File)this.targetFile);){
            int expectedNumberOfPages = ImageToPdfTransformerTest.calculateExpectedNumberOfPages(transformOptions, imageFile.firstPage(), imageFile.lastPage());
            Assertions.assertNotNull((Object)actualPdfDocument);
            Assertions.assertEquals((int)expectedNumberOfPages, (int)actualPdfDocument.getNumberOfPages());
        }
    }

    private static int calculateExpectedNumberOfPages(TransformOptions transformOptions, int firstPage, int lastPage) {
        int startPage = Optional.ofNullable(transformOptions.startPage).orElse(firstPage);
        int endPage = Math.min(Optional.ofNullable(transformOptions.endPage).orElse(lastPage), lastPage);
        return endPage - startPage + 1;
    }

    static Stream<TransformOptions> improperTransformOptions() {
        return Stream.of(TransformOptions.of(1, 0), TransformOptions.of(-1, 0), TransformOptions.of(0, -1));
    }

    @ParameterizedTest
    @MethodSource(value={"improperTransformOptions"})
    void testTransformTiffToPdf_withImproperOptions(TransformOptions transformOptions) {
        Assertions.assertThrows(IllegalArgumentException.class, () -> this.transformer.transform("image/tiff", "application/pdf", transformOptions.toMap(), sourceFile, this.targetFile, this.transformManager));
    }

    @Test
    void testTransformTiffToPdf_withInvalidStartPageOption() {
        Map<String, String> transformOptions = TransformOptions.none().toMap();
        transformOptions.put("startPage", "a");
        Assertions.assertThrows(IllegalArgumentException.class, () -> this.transformer.transform("image/tiff", "application/pdf", transformOptions, sourceFile, this.targetFile, this.transformManager));
    }

    @Test
    void testTransformTiffToPdf_withInvalidEndPageOption() {
        Map<String, String> transformOptions = TransformOptions.none().toMap();
        transformOptions.put("endPage", "z");
        Assertions.assertThrows(IllegalArgumentException.class, () -> this.transformer.transform("image/tiff", "application/pdf", transformOptions, sourceFile, this.targetFile, this.transformManager));
    }

    static Stream<Arguments> validPdfFormats() {
        return Stream.of(Arguments.of((Object[])new Object[]{"DEFAULT", new PDRectangle((float)sourceFileWidth, (float)sourceFileHeight)}), Arguments.of((Object[])new Object[]{"default", new PDRectangle((float)sourceFileWidth, (float)sourceFileHeight)}), Arguments.of((Object[])new Object[]{"A0", PDRectangle.A0}), Arguments.of((Object[])new Object[]{"a0", PDRectangle.A0}), Arguments.of((Object[])new Object[]{"A1", PDRectangle.A1}), Arguments.of((Object[])new Object[]{"A2", PDRectangle.A2}), Arguments.of((Object[])new Object[]{"A3", PDRectangle.A3}), Arguments.of((Object[])new Object[]{"A4", PDRectangle.A4}), Arguments.of((Object[])new Object[]{"A5", PDRectangle.A5}), Arguments.of((Object[])new Object[]{"A6", PDRectangle.A6}), Arguments.of((Object[])new Object[]{"A6", PDRectangle.A6}), Arguments.of((Object[])new Object[]{"LETTER", PDRectangle.LETTER}), Arguments.of((Object[])new Object[]{"letter", PDRectangle.LETTER}), Arguments.of((Object[])new Object[]{"LEGAL", PDRectangle.LEGAL}), Arguments.of((Object[])new Object[]{"legal", PDRectangle.LEGAL}));
    }

    static Stream<Arguments> validPdfOrientations() {
        return Stream.of(Arguments.of((Object[])new Object[]{"DEFAULT", ImageToPdfTransformerTest.unchangedRectangle()}), Arguments.of((Object[])new Object[]{"default", ImageToPdfTransformerTest.unchangedRectangle()}), Arguments.of((Object[])new Object[]{"PORTRAIT", ImageToPdfTransformerTest.rectangleRotatedIf((width, height) -> width.floatValue() > height.floatValue())}), Arguments.of((Object[])new Object[]{"portrait", ImageToPdfTransformerTest.rectangleRotatedIf((width, height) -> width.floatValue() > height.floatValue())}), Arguments.of((Object[])new Object[]{"LANDSCAPE", ImageToPdfTransformerTest.rectangleRotatedIf((width, height) -> height.floatValue() > width.floatValue())}), Arguments.of((Object[])new Object[]{"landscape", ImageToPdfTransformerTest.rectangleRotatedIf((width, height) -> height.floatValue() > width.floatValue())}));
    }

    static Stream<Arguments> validPdfFormatsAndOrientations() {
        return ArgumentsCartesianProduct.ofArguments(ImageToPdfTransformerTest.validPdfFormats(), ImageToPdfTransformerTest.validPdfOrientations());
    }

    @ParameterizedTest
    @MethodSource(value={"validPdfFormatsAndOrientations"})
    void testTransformImageToPDF_withVariousPdfFormatsAndOrientations(String pdfFormat, PDRectangle expectedPdfFormat, String pdfOrientation, BiFunction<Float, Float, PDRectangle> expectedPdfFormatRotator) throws Exception {
        TransformOptions transformOptions = TransformOptions.of(pdfFormat, pdfOrientation);
        this.transformer.transform("image/tiff", "application/pdf", transformOptions.toMap(), sourceFile, this.targetFile, this.transformManager);
        try (PDDocument actualPdfDocument = Loader.loadPDF((File)this.targetFile);){
            PDRectangle finalExpectedPdfFormat = expectedPdfFormatRotator.apply(Float.valueOf(expectedPdfFormat.getWidth()), Float.valueOf(expectedPdfFormat.getHeight()));
            Assertions.assertNotNull((Object)actualPdfDocument);
            Assertions.assertEquals((float)finalExpectedPdfFormat.getWidth(), (float)actualPdfDocument.getPage(0).getMediaBox().getWidth());
            Assertions.assertEquals((float)finalExpectedPdfFormat.getHeight(), (float)actualPdfDocument.getPage(0).getMediaBox().getHeight());
        }
    }

    @Test
    void testTransformImageToPDF_withInvalidPdfFormatAndUsingDefaultOne() throws Exception {
        TransformOptions transformOptions = TransformOptions.of("INVALID");
        this.transformer.transform("image/tiff", "application/pdf", transformOptions.toMap(), sourceFile, this.targetFile, this.transformManager);
        try (PDDocument actualPdfDocument = Loader.loadPDF((File)this.targetFile);){
            BufferedImage actualImage = ImageIO.read(sourceFile);
            Assertions.assertNotNull((Object)actualPdfDocument);
            Assertions.assertEquals((float)actualImage.getWidth(), (float)actualPdfDocument.getPage(0).getMediaBox().getWidth());
            Assertions.assertEquals((float)actualImage.getHeight(), (float)actualPdfDocument.getPage(0).getMediaBox().getHeight());
        }
    }

    @Test
    void testTransformImageToPDF_withInvalidPdfOrientationAndUsingDefaultOne() throws Exception {
        TransformOptions transformOptions = TransformOptions.of(null, "INVALID");
        this.transformer.transform("image/tiff", "application/pdf", transformOptions.toMap(), sourceFile, this.targetFile, this.transformManager);
        try (PDDocument actualPdfDocument = Loader.loadPDF((File)this.targetFile);){
            BufferedImage actualImage = ImageIO.read(sourceFile);
            Assertions.assertNotNull((Object)actualPdfDocument);
            Assertions.assertEquals((float)actualImage.getWidth(), (float)actualPdfDocument.getPage(0).getMediaBox().getWidth());
            Assertions.assertEquals((float)actualImage.getHeight(), (float)actualPdfDocument.getPage(0).getMediaBox().getHeight());
        }
    }

    static Stream<Arguments> imageFilesOfVariousSizeAndResolution() {
        return Stream.of(Arguments.of((Object[])new Object[]{ImageFile.of("MNT-24205.tiff", "image/tiff"), Float.valueOf(612.0f), Float.valueOf(792.0f)}), Arguments.of((Object[])new Object[]{ImageFile.of("459x594-1.tiff", "image/tiff"), Float.valueOf(459.0f), Float.valueOf(594.0f)}), Arguments.of((Object[])new Object[]{ImageFile.of("459x594-50.tif", "image/tiff"), Float.valueOf(660.0f), Float.valueOf(855.0f)}), Arguments.of((Object[])new Object[]{ImageFile.of("459x594-72.tif", "image/tiff"), Float.valueOf(459.0f), Float.valueOf(594.0f)}), Arguments.of((Object[])new Object[]{ImageFile.of("459x594-300.tif", "image/tiff"), Float.valueOf(110.0f), Float.valueOf(142.0f)}), Arguments.of((Object[])new Object[]{ImageFile.of("612x792-50.tif", "image/tiff"), Float.valueOf(881.0f), Float.valueOf(1140.0f)}), Arguments.of((Object[])new Object[]{ImageFile.of("612x792-72.tif", "image/tiff"), Float.valueOf(612.0f), Float.valueOf(792.0f)}), Arguments.of((Object[])new Object[]{ImageFile.of("612x792-300.tif", "image/tiff"), Float.valueOf(146.0f), Float.valueOf(190.0f)}), Arguments.of((Object[])new Object[]{ImageFile.of("765x990-50.tif", "image/tiff"), Float.valueOf(1101.0f), Float.valueOf(1425.0f)}), Arguments.of((Object[])new Object[]{ImageFile.of("765x990-72.tif", "image/tiff"), Float.valueOf(765.0f), Float.valueOf(990.0f)}), Arguments.of((Object[])new Object[]{ImageFile.of("765x990-300.tif", "image/tiff"), Float.valueOf(183.0f), Float.valueOf(237.0f)}));
    }

    @ParameterizedTest
    @MethodSource(value={"imageFilesOfVariousSizeAndResolution"})
    void testTransformTiffToPDF_withVariousImageSizes(ImageFile imageFile, float expectedWidth, float expectedHeight) throws Exception {
        TransformOptions transformOptions = TransformOptions.of("DEFAULT");
        File source = ImageToPdfTransformerTest.loadFile(imageFile.fileName);
        this.transformer.transform(imageFile.mimetype, "application/pdf", transformOptions.toMap(), source, this.targetFile, this.transformManager);
        try (PDDocument actualPdfDocument = Loader.loadPDF((File)this.targetFile);){
            Assertions.assertNotNull((Object)actualPdfDocument);
            Assertions.assertEquals((float)expectedWidth, (float)actualPdfDocument.getPage(0).getMediaBox().getWidth(), (String)"Pdf width");
            Assertions.assertEquals((float)expectedHeight, (float)actualPdfDocument.getPage(0).getMediaBox().getHeight(), (String)"Pdf height");
        }
    }

    @Test
    void testFindImageReaderForTiffFiles() {
        try (ImageInputStream imageInputStream = ImageIO.createImageInputStream(SOURCE_TIFF_FILE);){
            Method method = ImageToPdfTransformer.class.getDeclaredMethod("findImageReader", ImageInputStream.class, String.class, String.class);
            method.setAccessible(true);
            ImageReader imageReader = (ImageReader)method.invoke((Object)this.transformer, imageInputStream, "sample.tiff", "image/tiff");
            Assertions.assertNotNull((Object)imageReader, (String)"Image reader should not be null for TIFF file");
            Assertions.assertEquals((Object)"com.sun.imageio.plugins.tiff.TIFFImageReader", (Object)imageReader.getClass().getName(), (String)"ImageReader should be com.sun.imageio.plugins.tiff.TIFFImageReader");
        }
        catch (Exception e) {
            Assertions.fail((String)("Exception occurred: " + e.getMessage()));
        }
    }

    private static BiFunction<Float, Float, PDRectangle> unchangedRectangle() {
        return ImageToPdfTransformerTest.rectangleRotatedIf(null);
    }

    private static BiFunction<Float, Float, PDRectangle> rectangleRotatedIf(BiPredicate<Float, Float> predicate) {
        if (predicate == null) {
            return PDRectangle::new;
        }
        return (width, height) -> predicate.test((Float)width, (Float)height) ? new PDRectangle(height.floatValue(), width.floatValue()) : new PDRectangle(width.floatValue(), height.floatValue());
    }

    private static File loadFile(String fileName) {
        return new File(Objects.requireNonNull(ImageToPdfTransformerTest.class.getClassLoader().getResource(fileName)).getFile());
    }

    static {
        try {
            BufferedImage image = ImageIO.read(sourceFile);
            sourceFileWidth = image.getWidth();
            sourceFileHeight = image.getHeight();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private static class ImageFile {
        String fileName;
        String mimetype;
        int numberOfPages;

        private ImageFile(String fileName, String mimetype, int numberOfPages) {
            this.fileName = fileName;
            this.mimetype = mimetype;
            this.numberOfPages = numberOfPages;
        }

        public static ImageFile of(String fileName, String mimetype, int numberOfPages) {
            return new ImageFile(fileName, mimetype, numberOfPages);
        }

        public static ImageFile of(String fileName, String mimetype) {
            return ImageFile.of(fileName, mimetype, 1);
        }

        public int firstPage() {
            return 0;
        }

        public int lastPage() {
            return this.numberOfPages - 1;
        }

        public String toString() {
            return "ImageFile{fileName='" + this.fileName + "', mimetype='" + this.mimetype + "'}";
        }
    }

    private static class TransformOptions {
        Integer startPage;
        Integer endPage;
        String pdfFormat;
        String pdfOrientation;

        private TransformOptions(Integer startPage, Integer endPage, String pdfFormat, String pdfOrientation) {
            this.startPage = startPage;
            this.endPage = endPage;
            this.pdfFormat = pdfFormat;
            this.pdfOrientation = pdfOrientation;
        }

        public Map<String, String> toMap() {
            HashMap<String, String> transformOptions = new HashMap<String, String>();
            if (this.startPage != null) {
                transformOptions.put("startPage", this.startPage.toString());
            }
            if (this.endPage != null) {
                transformOptions.put("endPage", this.endPage.toString());
            }
            if (this.pdfFormat != null) {
                transformOptions.put("pdfFormat", this.pdfFormat);
            }
            if (this.pdfOrientation != null) {
                transformOptions.put("pdfOrientation", this.pdfOrientation);
            }
            return transformOptions;
        }

        public static TransformOptions of(Integer startPage, Integer endPage) {
            return new TransformOptions(startPage, endPage, null, null);
        }

        public static TransformOptions of(String pdfFormat) {
            return new TransformOptions(null, null, pdfFormat, null);
        }

        public static TransformOptions of(String pdfFormat, String pdfOrientation) {
            return new TransformOptions(null, null, pdfFormat, pdfOrientation);
        }

        public static TransformOptions none() {
            return TransformOptions.of(null);
        }

        public String toString() {
            return "TransformOption{startPage=" + this.startPage + ", endPage=" + this.endPage + "}";
        }
    }
}

