/*
 * #%L
 * Alfresco Repository
 * %%
 * Copyright (C) 2005 - 2022 Alfresco Software Limited
 * %%
 * This file is part of the Alfresco software.
 * If the software was purchased under a paid Alfresco license, the terms of
 * the paid license agreement will prevail.  Otherwise, the software is
 * provided under the following open source license terms:
 *
 * Alfresco is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Alfresco is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Alfresco. If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.alfresco.repo.event;

import java.net.URI;
import java.time.ZonedDateTime;

import org.alfresco.repo.event.extension.ExtensionAttributes;

/**
 * Event's attributes conforming to <a href="https://github.com/cloudevents/spec/blob/v1.0/spec.md">CloudEvents</a> <i>v1.0</i> specification.
 *
 * @author Jamal Kaabi-Mofrad
 */
public interface EventAttributes
{
    /**
     * @return The ID of the event, generated by the producer.
     */
    String getId();

    /**
     * @return The type of an event which has happened.
     */
    String getType();

    /**
     * @return The context in which an event happened. E.g. The instance of a DBP component that produced the event.
     */
    URI getSource();

    /**
     * @return The schema that {@code data} adheres to.
     */
    URI getDataschema();

    /**
     * @return The CloudEvents specification version
     */
    String getSpecversion();

    /**
     * @return The timestamp of when the occurrence happened.
     */
    ZonedDateTime getTime();

    /**
     * @return The content type of the data attribute adhering to the <a href="https://tools.ietf.org/html/rfc2046">RFC2046</a>
     */
    String getDatacontenttype();

    /**
     * @return <b>Optional</b> extension attributes that can be additionally added to the event. <br>
     *         See CloudEvents v1.0 specification <a href="https://github.com/cloudevents/spec/blob/v1.0/spec.md#extension-context-attributes">Extension Context Attributes</a> for more details.
     */
    default ExtensionAttributes getExtensionAttributes()
    {
        return null;
    }
}
