/*
 * Decompiled with CFR 0.152.
 */
package org.alfresco.transform.base.probes;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import org.alfresco.transform.base.fs.FileManager;
import org.alfresco.transform.base.logging.LogEntry;
import org.alfresco.transform.base.transform.TransformHandler;
import org.alfresco.transform.exceptions.TransformException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;

public class ProbeTransform {
    private final Logger logger = LoggerFactory.getLogger(ProbeTransform.class);
    private static final int AVERAGE_OVER_TRANSFORMS = 5;
    private final String sourceFilename;
    private final String sourceMimetype;
    private final String targetMimetype;
    private final Map<String, String> transformOptions;
    private final long minExpectedLength;
    private final long maxExpectedLength;
    private int livenessPercent;
    private long probeCount;
    private int transCount;
    private long normalTime;
    private long maxTime = Long.MAX_VALUE;
    private long nextTransformTime;
    private final boolean livenessTransformEnabled;
    private final long livenessTransformPeriod;
    private final long maxTransformCount;
    private long maxTransformTime;
    private final AtomicBoolean initialised = new AtomicBoolean(false);
    private final AtomicBoolean readySent = new AtomicBoolean(false);
    private final AtomicLong transformCount = new AtomicLong(0L);
    private final AtomicBoolean die = new AtomicBoolean(false);

    public int getLivenessPercent() {
        return this.livenessPercent;
    }

    public long getMaxTime() {
        return this.maxTime;
    }

    public ProbeTransform(String sourceFilename, String sourceMimetype, String targetMimetype, Map<String, String> transformOptions, long expectedLength, long plusOrMinus, int livenessPercent, long maxTransforms, long maxTransformSeconds, long livenessTransformPeriodSeconds) {
        this.sourceFilename = sourceFilename;
        this.sourceMimetype = sourceMimetype;
        this.targetMimetype = targetMimetype;
        this.transformOptions = new HashMap<String, String>(transformOptions);
        this.minExpectedLength = Math.max(0L, expectedLength - plusOrMinus);
        this.maxExpectedLength = expectedLength + plusOrMinus;
        this.livenessPercent = (int)this.getPositiveLongEnv("livenessPercent", livenessPercent);
        this.maxTransformCount = this.getPositiveLongEnv("maxTransforms", maxTransforms);
        this.maxTransformTime = this.getPositiveLongEnv("maxTransformSeconds", maxTransformSeconds) * 1000L;
        this.livenessTransformPeriod = this.getPositiveLongEnv("livenessTransformPeriodSeconds", livenessTransformPeriodSeconds) * 1000L;
        this.livenessTransformEnabled = this.getBooleanEnvVar("livenessTransformEnabled", false);
    }

    private boolean getBooleanEnvVar(String name, boolean defaultValue) {
        try {
            return Boolean.parseBoolean(System.getenv(name));
        }
        catch (Exception exception) {
            return defaultValue;
        }
    }

    private long getPositiveLongEnv(String name, long defaultValue) {
        long l = -1L;
        String env = System.getenv(name);
        if (env != null) {
            try {
                l = Long.parseLong(env);
            }
            catch (NumberFormatException numberFormatException) {
                // empty catch block
            }
        }
        if (l <= 0L) {
            l = defaultValue;
        }
        this.logger.trace("Probe: {}={}", (Object)name, (Object)l);
        return l;
    }

    public String doTransformOrNothing(boolean isLiveProbe, TransformHandler transformHandler) {
        this.checkMaxTransformTimeAndCount(isLiveProbe);
        ++this.probeCount;
        if (isLiveProbe && !this.livenessTransformEnabled) {
            return this.doNothing(true);
        }
        String result = isLiveProbe && this.livenessTransformPeriod > 0L && (this.transCount <= 5 || this.nextTransformTime < System.currentTimeMillis()) || !this.initialised.get() ? this.doTransform(isLiveProbe, transformHandler) : this.doNothing(isLiveProbe);
        this.checkMaxTransformTimeAndCount(isLiveProbe);
        return result;
    }

    private String doNothing(boolean isLiveProbe) {
        String probeMessage = this.getProbeMessage(isLiveProbe);
        String message = "Success - No transform.";
        if (!isLiveProbe && !this.readySent.getAndSet(true)) {
            this.logger.trace("{}{}", (Object)probeMessage, (Object)message);
        }
        return message;
    }

    private String doTransform(boolean isLiveProbe, TransformHandler transformHandler) {
        long start = System.currentTimeMillis();
        if (this.nextTransformTime != 0L) {
            do {
                this.nextTransformTime += this.livenessTransformPeriod;
            } while (this.nextTransformTime < start);
        }
        File sourceFile = this.getSourceFile(isLiveProbe);
        File targetFile = this.getTargetFile();
        transformHandler.handleProbeRequest(this.sourceMimetype, this.targetMimetype, this.transformOptions, sourceFile, targetFile, this);
        long time = System.currentTimeMillis() - start;
        String message = "Transform " + time + "ms";
        this.checkTargetFile(targetFile, isLiveProbe);
        this.recordTransformTime(time);
        this.calculateMaxTime(time, isLiveProbe);
        if (time > this.maxTime) {
            throw new TransformException(HttpStatus.INTERNAL_SERVER_ERROR, this.getMessagePrefix(isLiveProbe) + message + " which is more than " + this.livenessPercent + "% slower than the normal value of " + this.normalTime + "ms");
        }
        this.initialised.set(true);
        return this.getProbeMessage(isLiveProbe) + "Success - " + message;
    }

    private void checkMaxTransformTimeAndCount(boolean isLiveProbe) {
        if (this.die.get()) {
            throw new TransformException(HttpStatus.TOO_MANY_REQUESTS, this.getMessagePrefix(isLiveProbe) + "Transformer requested to die. A transform took longer than " + this.maxTransformTime / 1000L + " seconds");
        }
        if (this.maxTransformCount > 0L && this.transformCount.get() > this.maxTransformCount) {
            throw new TransformException(HttpStatus.TOO_MANY_REQUESTS, this.getMessagePrefix(isLiveProbe) + "Transformer requested to die. It has performed more than " + this.maxTransformCount + " transformations");
        }
    }

    private File getSourceFile(boolean isLiveProbe) {
        File sourceFile = FileManager.TempFileProvider.createTempFile("probe_source_", "_" + this.sourceFilename);
        try (InputStream inputStream = this.getClass().getResourceAsStream("/" + this.sourceFilename);){
            Files.copy(inputStream, sourceFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
        }
        catch (IOException e) {
            throw new TransformException(HttpStatus.INSUFFICIENT_STORAGE, this.getMessagePrefix(isLiveProbe) + "Failed to store the source file", (Throwable)e);
        }
        long length = sourceFile.length();
        LogEntry.setSource(sourceFile.getName(), length);
        return sourceFile;
    }

    private File getTargetFile() {
        File targetFile = FileManager.TempFileProvider.createTempFile("probe_target_", "_" + this.sourceFilename);
        LogEntry.setTarget(targetFile.getName());
        return targetFile;
    }

    public void recordTransformTime(long time) {
        if (this.maxTransformTime > 0L && time > this.maxTransformTime) {
            this.die.set(true);
        }
    }

    public void calculateMaxTime(long time, boolean isLiveProbe) {
        if (this.transCount <= 5) {
            String message = this.getMessagePrefix(isLiveProbe) + "Success - Transform " + time + "ms";
            if (++this.transCount > 1) {
                this.normalTime = (this.normalTime * (long)(this.transCount - 2) + time) / (long)(this.transCount - 1);
                this.maxTime = this.normalTime * (long)(this.livenessPercent + 100) / 100L;
                if (!isLiveProbe && !this.readySent.getAndSet(true) || this.transCount > 5) {
                    this.nextTransformTime = System.currentTimeMillis() + this.livenessTransformPeriod;
                    this.logger.trace("{} - {}ms+{}%={}ms", new Object[]{message, this.normalTime, this.livenessPercent, this.maxTime});
                }
            } else if (!isLiveProbe && !this.readySent.getAndSet(true)) {
                this.logger.trace(message);
            }
        }
    }

    private void checkTargetFile(File targetFile, boolean isLiveProbe) {
        String probeMessage = this.getProbeMessage(isLiveProbe);
        if (!targetFile.exists() || !targetFile.isFile()) {
            throw new TransformException(HttpStatus.INTERNAL_SERVER_ERROR, probeMessage + "Target File \"" + targetFile.getAbsolutePath() + "\" did not exist");
        }
        long length = targetFile.length();
        targetFile.delete();
        if (length < this.minExpectedLength || length > this.maxExpectedLength) {
            throw new TransformException(HttpStatus.INTERNAL_SERVER_ERROR, probeMessage + "Target File \"" + targetFile.getAbsolutePath() + "\" was the wrong size (" + length + "). Needed to be between " + this.minExpectedLength + " and " + this.maxExpectedLength);
        }
    }

    private String getMessagePrefix(boolean isLiveProbe) {
        return Long.toString(this.probeCount) + " " + this.getProbeMessage(isLiveProbe);
    }

    private String getProbeMessage(boolean isLiveProbe) {
        return isLiveProbe ? "Live Probe: " : "Ready Probe: ";
    }

    public void incrementTransformerCount() {
        this.transformCount.incrementAndGet();
    }

    public void setLivenessPercent(int livenessPercent) {
        this.livenessPercent = livenessPercent;
    }

    public long getNormalTime() {
        return this.normalTime;
    }

    public void resetForTesting() {
        this.probeCount = 0L;
        this.transCount = 0;
        this.normalTime = 0L;
        this.maxTime = Long.MAX_VALUE;
        this.nextTransformTime = 0L;
        this.initialised.set(false);
        this.readySent.set(false);
        this.transformCount.set(0L);
        this.die.set(false);
    }
}

