/*
 * Decompiled with CFR 0.152.
 */
package org.alfresco.repo.content.caching.quota;

import java.io.File;
import java.util.concurrent.atomic.AtomicLong;
import org.alfresco.repo.content.caching.ContentCacheImpl;
import org.alfresco.repo.content.caching.cleanup.CachedContentCleaner;
import org.alfresco.repo.content.caching.quota.QuotaManagerStrategy;
import org.alfresco.repo.content.caching.quota.UsageTracker;
import org.alfresco.repo.content.filestore.FileContentReader;
import org.alfresco.repo.content.filestore.FileContentWriter;
import org.alfresco.util.PropertyCheck;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Required;

public class StandardQuotaStrategy
implements QuotaManagerStrategy,
UsageTracker {
    private static final String CACHE_USAGE_FILENAME = "cache-usage.txt";
    private static final Log log = LogFactory.getLog(StandardQuotaStrategy.class);
    private static final long DEFAULT_DISK_USAGE_ESTIMATE = 0L;
    private int panicThresholdPct = 90;
    private int cleanThresholdPct = 80;
    private int targetUsagePct = 70;
    private long maxUsageBytes = 0L;
    private long normalCleanThresholdSec = 0L;
    private AtomicLong currentUsageBytes = new AtomicLong(0L);
    private AtomicLong lastCleanupStart = new AtomicLong(0L);
    private CachedContentCleaner cleaner;
    private ContentCacheImpl cache;
    private int maxFileSizeMB = 0;

    public void init() {
        if (log.isDebugEnabled()) {
            log.debug((Object)"Starting quota strategy.");
        }
        PropertyCheck.mandatory((Object)this, (String)"cleaner", (Object)this.cleaner);
        PropertyCheck.mandatory((Object)this, (String)"cache", (Object)this.cache);
        if (this.maxUsageBytes < 0xA00000L && log.isWarnEnabled()) {
            log.warn((Object)("Low maxUsageBytes of " + this.maxUsageBytes + "bytes - did you mean to specify in MB?"));
        }
        this.loadDiskUsage();
        this.lastCleanupStart.set(System.currentTimeMillis() - this.normalCleanThresholdSec);
        this.signalCleanerStart("quota (init)");
    }

    public void shutdown() {
        if (log.isDebugEnabled()) {
            log.debug((Object)"Shutting down quota strategy.");
        }
        this.saveDiskUsage();
    }

    private void loadDiskUsage() {
        File usageFile = new File(this.cache.getCacheRoot(), CACHE_USAGE_FILENAME);
        if (!usageFile.exists()) {
            this.setCurrentUsageBytes(0L);
            if (log.isInfoEnabled()) {
                log.info((Object)("No previous usage file found (" + usageFile + ") so assuming: " + this.getCurrentUsageBytes() + " bytes."));
            }
        } else {
            FileContentReader reader = new FileContentReader(usageFile);
            String usageStr = reader.getContentString();
            long usage = Long.parseLong(usageStr);
            this.currentUsageBytes.set(usage);
            if (log.isInfoEnabled()) {
                log.info((Object)("Using last known disk usage estimate: " + this.getCurrentUsageBytes()));
            }
        }
    }

    private void saveDiskUsage() {
        File usageFile = new File(this.cache.getCacheRoot(), CACHE_USAGE_FILENAME);
        FileContentWriter writer = new FileContentWriter(usageFile);
        writer.putContent(this.currentUsageBytes.toString());
    }

    @Override
    public boolean beforeWritingCacheFile(long contentSizeBytes) {
        long maxFileSizeBytes = this.getMaxFileSizeBytes();
        if (maxFileSizeBytes > 0L && contentSizeBytes > maxFileSizeBytes) {
            if (log.isDebugEnabled()) {
                log.debug((Object)("File too large (" + contentSizeBytes + " bytes, max allowed is " + this.getMaxFileSizeBytes() + ") - vetoing disk write."));
            }
            return false;
        }
        if (this.usageWillReach(this.panicThresholdPct, contentSizeBytes)) {
            if (log.isDebugEnabled()) {
                log.debug((Object)("Panic threshold reached (" + this.panicThresholdPct + "%) - vetoing disk write and starting cached content cleaner."));
            }
            this.signalCleanerStart("quota (panic threshold)");
            return false;
        }
        return true;
    }

    @Override
    public boolean afterWritingCacheFile(long contentSizeBytes) {
        boolean keepNewFile = true;
        long maxFileSizeBytes = this.getMaxFileSizeBytes();
        if (maxFileSizeBytes > 0L && contentSizeBytes > maxFileSizeBytes) {
            keepNewFile = false;
        } else {
            this.addUsageBytes(contentSizeBytes);
        }
        if (this.getCurrentUsageBytes() >= this.maxUsageBytes) {
            if (log.isDebugEnabled()) {
                log.debug((Object)("Usage has reached or exceeded quota limit, limit: " + this.maxUsageBytes + " bytes, current usage: " + this.getCurrentUsageBytes() + " bytes."));
            }
            this.signalAggressiveCleanerStart("quota (limit reached)");
        } else if (this.usageHasReached(this.cleanThresholdPct)) {
            if (log.isDebugEnabled()) {
                log.debug((Object)("Usage has reached " + this.cleanThresholdPct + "% - starting cached content cleaner."));
            }
            this.signalCleanerStart("quota (clean threshold)");
        }
        return keepNewFile;
    }

    private void signalCleanerStart(String reason, boolean aggressive) {
        if (aggressive) {
            long targetReductionBytes = (long)((double)this.targetUsagePct / 100.0 * (double)this.maxUsageBytes);
            this.cleaner.executeAggressive(reason, targetReductionBytes);
        } else {
            long timePassedFromLastClean = System.currentTimeMillis() - this.lastCleanupStart.get();
            if (timePassedFromLastClean < this.normalCleanThresholdSec * 1000L) {
                if (log.isDebugEnabled()) {
                    log.debug((Object)("Skipping a normal clean as it is too soon. The last cleanup was run " + (float)timePassedFromLastClean / 1000.0f + " seconds ago."));
                }
            } else {
                this.lastCleanupStart.set(System.currentTimeMillis());
                this.cleaner.execute(reason);
            }
        }
    }

    private void signalCleanerStart(String reason) {
        this.signalCleanerStart(reason, false);
    }

    private void signalAggressiveCleanerStart(String reason) {
        this.signalCleanerStart(reason, true);
    }

    private boolean usageWillReach(int threshold, long contentSize) {
        long potentialUsage = this.getCurrentUsageBytes() + contentSize;
        double pctOfMaxAllowed = (double)potentialUsage / (double)this.maxUsageBytes * 100.0;
        return pctOfMaxAllowed >= (double)threshold;
    }

    private boolean usageHasReached(int threshold) {
        return this.usageWillReach(threshold, 0L);
    }

    public void setMaxUsageMB(long maxUsageMB) {
        this.setMaxUsageBytes(maxUsageMB * 0x100000L);
    }

    public void setMaxUsageBytes(long maxUsageBytes) {
        this.maxUsageBytes = maxUsageBytes;
    }

    public void setPanicThresholdPct(int panicThresholdPct) {
        this.panicThresholdPct = panicThresholdPct;
    }

    public void setCleanThresholdPct(int cleanThresholdPct) {
        this.cleanThresholdPct = cleanThresholdPct;
    }

    public void setTargetUsagePct(int targetUsagePct) {
        this.targetUsagePct = targetUsagePct;
    }

    public void setNormalCleanThresholdSec(long normalCleanThresholdSec) {
        this.normalCleanThresholdSec = normalCleanThresholdSec;
    }

    @Required
    public void setCache(ContentCacheImpl cache) {
        this.cache = cache;
    }

    @Required
    public void setCleaner(CachedContentCleaner cleaner) {
        this.cleaner = cleaner;
    }

    @Override
    public long getCurrentUsageBytes() {
        return this.currentUsageBytes.get();
    }

    public double getCurrentUsageMB() {
        return (double)this.getCurrentUsageBytes() / 1048576.0;
    }

    public long getMaxUsageBytes() {
        return this.maxUsageBytes;
    }

    public long getMaxUsageMB() {
        return this.maxUsageBytes / 0x100000L;
    }

    public int getMaxFileSizeMB() {
        return this.maxFileSizeMB;
    }

    protected long getMaxFileSizeBytes() {
        return (long)this.maxFileSizeMB * 0x100000L;
    }

    public void setMaxFileSizeMB(int maxFileSizeMB) {
        this.maxFileSizeMB = maxFileSizeMB;
    }

    @Override
    public long addUsageBytes(long sizeDelta) {
        long newUsage = this.currentUsageBytes.addAndGet(sizeDelta);
        if (log.isDebugEnabled()) {
            log.debug((Object)String.format("Disk usage changed by %d to %d bytes", sizeDelta, newUsage));
        }
        return newUsage;
    }

    @Override
    public void setCurrentUsageBytes(long newDiskUsage) {
        if (log.isInfoEnabled()) {
            log.info((Object)String.format("Setting disk usage to %d bytes", newDiskUsage));
        }
        this.currentUsageBytes.set(newDiskUsage);
    }
}

