/*
 * Decompiled with CFR 0.152.
 */
package org.alfresco.repo.bulkimport.impl;

import java.io.IOException;
import java.io.Serializable;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.alfresco.repo.bulkimport.AnalysedDirectory;
import org.alfresco.repo.bulkimport.DirectoryAnalyser;
import org.alfresco.repo.bulkimport.ImportFilter;
import org.alfresco.repo.bulkimport.ImportableItem;
import org.alfresco.repo.bulkimport.MetadataLoader;
import org.alfresco.repo.bulkimport.impl.BulkImportStatusImpl;
import org.alfresco.repo.bulkimport.impl.FileUtils;
import org.alfresco.repo.dictionary.constraint.NameChecker;
import org.alfresco.service.cmr.dictionary.Constraint;
import org.alfresco.service.cmr.dictionary.ConstraintDefinition;
import org.alfresco.service.cmr.dictionary.ConstraintException;
import org.alfresco.service.cmr.dictionary.DictionaryService;
import org.alfresco.service.cmr.dictionary.PropertyDefinition;
import org.alfresco.service.namespace.QName;
import org.alfresco.util.ISO8601DateFormat;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.extensions.surf.exception.PlatformRuntimeException;

public class DirectoryAnalyserImpl
implements DirectoryAnalyser {
    private static final Log log = LogFactory.getLog(DirectoryAnalyserImpl.class);
    private static final Pattern VERSION_SUFFIX_PATTERN = Pattern.compile(".+\\.v([0-9]+)\\z");
    private MetadataLoader metadataLoader;
    private BulkImportStatusImpl importStatus;
    private List<ImportFilter> importFilters;
    private NameChecker nameChecker;
    private DictionaryService dictionaryService;

    public DirectoryAnalyserImpl(MetadataLoader metadataLoader, BulkImportStatusImpl importStatus, List<ImportFilter> importFilters, NameChecker nameChecker) {
        this.metadataLoader = metadataLoader;
        this.importStatus = importStatus;
        this.importFilters = importFilters;
        this.nameChecker = nameChecker;
    }

    public DirectoryAnalyserImpl() {
    }

    public void setDictionaryService(DictionaryService dictionaryService) {
        this.dictionaryService = dictionaryService;
    }

    public void setNameChecker(NameChecker nameChecker) {
        this.nameChecker = nameChecker;
    }

    public void setMetadataLoader(MetadataLoader metadataLoader) {
        this.metadataLoader = metadataLoader;
    }

    public void setImportStatus(BulkImportStatusImpl status) {
        this.importStatus = status;
    }

    public final void setImportFilters(List<ImportFilter> importFilters) {
        this.importFilters = importFilters != null ? importFilters : new ArrayList<ImportFilter>();
    }

    protected boolean shouldFilter(ImportableItem importableItem) {
        boolean filterImportableItem = false;
        if (this.importFilters != null && this.importFilters.size() > 0) {
            for (ImportFilter filter : this.importFilters) {
                if (!filter.shouldFilter(importableItem)) continue;
                filterImportableItem = true;
                break;
            }
        }
        return filterImportableItem;
    }

    @Override
    public AnalysedDirectory analyseDirectory(ImportableItem directory, DirectoryStream.Filter<Path> filter) {
        ImportableItem importableItem;
        Path directoryFile = directory.getHeadRevision().getContentFile();
        AnalysedDirectory result = new AnalysedDirectory(this.listFiles(directoryFile, filter));
        if (log.isDebugEnabled()) {
            log.debug((Object)("Analysing directory " + FileUtils.getFileName(directoryFile) + "..."));
        }
        for (Path file : result.getOriginalPaths()) {
            String isoDate = null;
            try {
                isoDate = ISO8601DateFormat.format((Date)new Date(Files.getLastModifiedTime(file, LinkOption.NOFOLLOW_LINKS).toMillis()));
                ISO8601DateFormat.parse((String)isoDate);
            }
            catch (IOException | PlatformRuntimeException e) {
                log.warn((Object)("Failed to convert date " + isoDate + " to string for " + file.getFileName()), e);
                this.importStatus.incrementNumberOfUnreadableEntries();
                continue;
            }
            if (log.isTraceEnabled()) {
                log.trace((Object)("Scanning file " + FileUtils.getFileName(file) + "..."));
            }
            if (Files.isReadable(file)) {
                try {
                    this.nameChecker.evaluate((Object)file.getFileName().toString());
                }
                catch (ConstraintException constraintException) {
                    if (log.isWarnEnabled()) {
                        log.warn((Object)("Skipping file with invalid name: '" + FileUtils.getFileName(file) + "'."));
                    }
                    this.importStatus.incrementNumberOfUnreadableEntries();
                    continue;
                }
                if (this.isVersionFile(file)) {
                    this.addVersionFile(directory, result, file);
                    this.importStatus.incrementNumberOfFilesScanned();
                    continue;
                }
                if (this.isMetadataFile(file)) {
                    this.addMetadataFile(directory, result, file);
                    this.importStatus.incrementNumberOfFilesScanned();
                    continue;
                }
                boolean isDirectory = this.addParentFile(directory, result, file);
                if (isDirectory) {
                    this.importStatus.incrementNumberOfFoldersScanned();
                    continue;
                }
                this.importStatus.incrementNumberOfFilesScanned();
                continue;
            }
            if (log.isWarnEnabled()) {
                log.warn((Object)("Skipping unreadable file '" + FileUtils.getFileName(file) + "'."));
            }
            this.importStatus.incrementNumberOfUnreadableEntries();
        }
        Iterator<ImportableItem> iter = result.getImportableItems().iterator();
        while (iter.hasNext()) {
            importableItem = iter.next();
            if (importableItem.isValid() && this.isMetadataValid(importableItem)) continue;
            iter.remove();
        }
        iter = result.getImportableDirectories().iterator();
        while (iter.hasNext()) {
            importableItem = iter.next();
            if (importableItem.isValid()) continue;
            iter.remove();
        }
        if (log.isDebugEnabled()) {
            log.debug((Object)("Finished analysing directory " + FileUtils.getFileName(directoryFile) + "."));
        }
        return result;
    }

    private List<Path> listFiles(Path sourceDirectory, DirectoryStream.Filter<Path> filter) {
        ArrayList<Path> files = new ArrayList<Path>();
        try {
            Throwable throwable = null;
            Object var5_7 = null;
            try (DirectoryStream<Path> paths = filter != null ? Files.newDirectoryStream(sourceDirectory, filter) : Files.newDirectoryStream(sourceDirectory);){
                Iterator<Path> it = paths.iterator();
                while (it.hasNext()) {
                    files.add(it.next());
                }
            }
            catch (Throwable throwable2) {
                if (throwable == null) {
                    throwable = throwable2;
                } else if (throwable != throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
        }
        catch (IOException e) {
            log.error((Object)e.getMessage());
        }
        return files;
    }

    private boolean isMetadataValid(ImportableItem importableItem) {
        if (!importableItem.getHeadRevision().metadataFileExists()) {
            return true;
        }
        if (this.metadataLoader != null) {
            MetadataLoader.Metadata result = new MetadataLoader.Metadata();
            this.metadataLoader.loadMetadata(importableItem.getHeadRevision(), result);
            Map<QName, Serializable> metadataProperties = result.getProperties();
            for (QName propertyName : metadataProperties.keySet()) {
                PropertyDefinition propDef = this.dictionaryService.getProperty(propertyName);
                if (propDef == null) continue;
                for (ConstraintDefinition constraintDef : propDef.getConstraints()) {
                    Constraint constraint = constraintDef.getConstraint();
                    if (constraint == null) continue;
                    try {
                        constraint.evaluate((Object)metadataProperties.get(propertyName));
                    }
                    catch (ConstraintException e) {
                        if (log.isWarnEnabled()) {
                            log.warn((Object)("Skipping file '" + FileUtils.getFileName(importableItem.getHeadRevision().getContentFile()) + "' with invalid metadata: '" + FileUtils.getFileName(importableItem.getHeadRevision().getMetadataFile()) + "'."), (Throwable)e);
                        }
                        return false;
                    }
                }
            }
        }
        return true;
    }

    private boolean isVersionFile(Path file) {
        Matcher matcher = VERSION_SUFFIX_PATTERN.matcher(file.getFileName().toString());
        return matcher.matches();
    }

    private boolean isMetadataFile(Path file) {
        boolean result = false;
        if (this.metadataLoader != null) {
            String name = file.getFileName().toString();
            result = name.endsWith(".metadata." + this.metadataLoader.getMetadataFileExtension());
        }
        return result;
    }

    private void addVersionFile(ImportableItem parent, AnalysedDirectory analysedDirectory, Path versionFile) {
        Path parentContentFile = this.getParentOfVersionFile(versionFile);
        boolean isContentVersion = false;
        if (this.isMetadataFile(parentContentFile)) {
            parentContentFile = this.getParentOfMetadatafile(parentContentFile);
            isContentVersion = false;
        } else {
            isContentVersion = true;
        }
        ImportableItem importableItem = this.findOrCreateImportableItem(parent, analysedDirectory, parentContentFile);
        int version = this.getVersionNumber(versionFile);
        ImportableItem.VersionedContentAndMetadata versionEntry = this.findOrCreateVersionEntry(importableItem, version);
        if (isContentVersion) {
            versionEntry.setContentFile(versionFile);
        } else {
            versionEntry.setMetadataFile(versionFile);
        }
    }

    private void addMetadataFile(ImportableItem parent, AnalysedDirectory analysedDirectory, Path metadataFile) {
        Path parentContentfile = this.getParentOfMetadatafile(metadataFile);
        ImportableItem importableItem = this.findOrCreateImportableItem(parent, analysedDirectory, parentContentfile);
        importableItem.getHeadRevision().setMetadataFile(metadataFile);
    }

    private boolean addParentFile(ImportableItem parent, AnalysedDirectory analysedDirectory, Path contentFile) {
        ImportableItem importableItem = this.findOrCreateImportableItem(parent, analysedDirectory, contentFile);
        importableItem.getHeadRevision().setContentFile(contentFile);
        return importableItem.getHeadRevision().getContentFileType() == ImportableItem.FileType.DIRECTORY;
    }

    private ImportableItem findOrCreateImportableItem(ImportableItem parent, AnalysedDirectory analysedDirectory, Path contentFile) {
        ImportableItem result = this.findImportableItem(analysedDirectory, contentFile);
        if (result == null) {
            result = new ImportableItem();
            result.setParent(parent);
            result.getHeadRevision().setContentFile(contentFile);
            if (!this.shouldFilter(result)) {
                analysedDirectory.addImportableItem(result);
            }
        }
        return result;
    }

    private ImportableItem findImportableItem(AnalysedDirectory analysedDirectory, Path contentFile) {
        ImportableItem result = null;
        if (contentFile == null) {
            throw new IllegalStateException("Cannot call findOrCreateImportableItem with null key");
        }
        result = analysedDirectory.findImportableItem(contentFile);
        return result;
    }

    private ImportableItem.VersionedContentAndMetadata findOrCreateVersionEntry(ImportableItem importableItem, int version) {
        ImportableItem.VersionedContentAndMetadata result = this.findVersionEntry(importableItem, version);
        if (result == null) {
            ImportableItem importableItem2 = importableItem;
            importableItem2.getClass();
            result = new ImportableItem.VersionedContentAndMetadata(importableItem2, version);
            importableItem.addVersionEntry(result);
        }
        return result;
    }

    private ImportableItem.VersionedContentAndMetadata findVersionEntry(ImportableItem importableItem, int version) {
        ImportableItem.VersionedContentAndMetadata result = null;
        if (importableItem.hasVersionEntries()) {
            for (ImportableItem.VersionedContentAndMetadata versionEntry : importableItem.getVersionEntries()) {
                if (version != versionEntry.getVersion()) continue;
                result = versionEntry;
                break;
            }
        }
        return result;
    }

    private int getVersionNumber(Path versionFile) {
        int result = -1;
        if (!this.isVersionFile(versionFile)) {
            throw new IllegalStateException(String.valueOf(FileUtils.getFileName(versionFile)) + " is not a version file.");
        }
        Matcher matcher = VERSION_SUFFIX_PATTERN.matcher(versionFile.getFileName().toString());
        String versionStr = null;
        if (!matcher.matches()) {
            throw new IllegalStateException("");
        }
        versionStr = matcher.group(1);
        result = Integer.parseInt(versionStr);
        return result;
    }

    private Path getParentOfVersionFile(Path versionFile) {
        Path result = null;
        if (!this.isVersionFile(versionFile)) {
            throw new IllegalStateException(String.valueOf(FileUtils.getFileName(versionFile)) + " is not a version file.");
        }
        String parentFilename = versionFile.getFileName().toString().replaceFirst("\\.v([0-9]+)\\z", "");
        result = versionFile.getParent().resolve(parentFilename);
        return result;
    }

    private Path getParentOfMetadatafile(Path metadataFile) {
        Path result = null;
        if (!this.isMetadataFile(metadataFile)) {
            throw new IllegalStateException(String.valueOf(FileUtils.getFileName(metadataFile)) + " is not a metadata file.");
        }
        String name = metadataFile.getFileName().toString();
        String contentName = name.substring(0, name.length() - (".metadata." + this.metadataLoader.getMetadataFileExtension()).length());
        result = metadataFile.getParent().resolve(contentName);
        return result;
    }
}

