/*
 * Copyright 2005-2022 Alfresco Software, Ltd.  All rights reserved.
 *
 * This file is part of a proprietary Alfresco module.
 *
 * License rights for this program are granted under the terms of the "Alfresco
 * Component License", which defines the permitted uses of the module.
 * License terms can be found in the file license.txt distributed with this module.
 */
package org.alfresco.module.aosmodule.dispatch;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.List;

import jakarta.servlet.ServletConfig;
import jakarta.servlet.ServletException;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;

import org.alfresco.officeservices.GetWebCollectionWebDescription;
import org.alfresco.officeservices.GetWebWebDescription;
import org.alfresco.officeservices.StandardWebsService;
import org.alfresco.officeservices.UserData;
import org.alfresco.officeservices.WebsGetContentTypeContentType;
import org.alfresco.officeservices.WebsGetContentTypesContentType;
import org.alfresco.officeservices.exceptions.AuthenticationRequiredException;
import org.alfresco.officeservices.protocol.SimpleSoapParser;
import org.alfresco.officeservices.protocol.SoapParameter;
import org.slf4j.LoggerFactory;
import org.slf4j.Logger;

public class WebsService extends StandardWebsService
{

	private static final long serialVersionUID = -6294184506392454609L;

	protected List<String> servicePrefixes = new ArrayList<String>();
    
    private Logger logger = LoggerFactory.getLogger(WebsService.class);

    public void init(ServletConfig servletConfig_p) throws ServletException
    {
        super.init(servletConfig_p);
        // read list of service names
        String spaceSeparatedEncodedServices = getServletContext().getInitParameter("org.alfresco.module.aosmodule.dispatch.SERVICES");
        if(spaceSeparatedEncodedServices != null)
        {
            for(String encodedContext : spaceSeparatedEncodedServices.split(" "))
            {
                try
                {
                    servicePrefixes.add(URLDecoder.decode(encodedContext, "UTF-8"));
                }
                catch (UnsupportedEncodingException e)
                {
                    logger.error("Unsupported encoding",e);
                }
            }
        }
        // and finally add the server root as service prefix
        servicePrefixes.add("");
    }

	@Override
    public List<String> getServicePrefixes(SimpleSoapParser parser, HttpServletRequest request)
    {
        return servicePrefixes;
    }
    
    protected String getServerUrl(HttpServletRequest request)
    {
        String protocol = request.isSecure() ? "https://" : "http://";
        int defaultPort = request.isSecure() ? 443 : 80;
        String portString = (request.getLocalPort() != defaultPort) ? ":" + Integer.toString(request.getLocalPort()) : "";
        return protocol + request.getServerName() + portString;
    }

	@Override
    public List<GetWebCollectionWebDescription> getWebCollection(SimpleSoapParser parser, HttpServletRequest request)
    {
        String serverUrl = getServerUrl(request);
        List<GetWebCollectionWebDescription> webCollection = new ArrayList<GetWebCollectionWebDescription>(servicePrefixes.size()-1);
        for(String servicePrefix : servicePrefixes)
        {
            String webUrlForService = serverUrl + servicePrefix;
            String webTitle = servicePrefix.length() > 1 && servicePrefix.startsWith("/") ? servicePrefix.substring(1) : servicePrefix;
            webCollection.add(new GetWebCollectionWebDescription(webTitle,webUrlForService));
        }
        return webCollection;
    }

	@Override
	protected GetWebWebDescription getWebResult(UserData userData, String webUrl, SimpleSoapParser parser, HttpServletRequest request)
	{
        String serverUrl = getServerUrl(request);
        for(String servicePrefix : servicePrefixes)
        {
            String webUrlForService = serverUrl + servicePrefix;
            if(webUrl.equals(webUrlForService))
            {
                String webTitle = servicePrefix.length() > 1 && servicePrefix.startsWith("/") ? servicePrefix.substring(1) : servicePrefix;
                return new GetWebWebDescription(webTitle,webUrlForService,"",GetWebWebDescription.LCID_ENUS,"");
            }
        }
		return null;
	}

    protected class WebsServiceUserData implements UserData
    {

        public String getUsername()
        {
            return "";
        }

    }

    @Override
    public WebsGetContentTypesContentType[] getContentTypes(SimpleSoapParser parser, HttpServletRequest request)
    {
        return null;
    }

    @Override
    protected WebsGetContentTypeContentType getContentType(UserData userData, String contentTypeId, HttpServletRequest request) throws AuthenticationRequiredException
    {
        return null;
    }

    protected UserData userData = new WebsServiceUserData();

	@Override
    public UserData negotiateAuthentication(HttpServletRequest arg0, HttpServletResponse arg1) throws IOException
    {
        return userData;
    }

	@Override
    public void invalidateAuthentication(UserData userData, HttpServletRequest request, HttpServletResponse response) throws IOException
    {
        // nothing to do here
    }

	@Override
    public void requestAuthentication(HttpServletRequest request, HttpServletResponse response) throws IOException
    {
        // nothing to do here
    }

    @Override
    protected boolean updateContentTypeXmlDocument(UserData userData, String contentTypeId, SoapParameter newDocument, HttpServletRequest request) throws AuthenticationRequiredException
    {
        //  content type  XSN update is not implemented
        return false;
    }

}