package org.alfresco.aos.dsl;

import static org.alfresco.utility.report.log.Step.STEP;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.alfresco.aos.AOSWrapper;
import org.alfresco.utility.data.DataContent;
import org.alfresco.utility.dsl.DSLAssertion;
import org.alfresco.utility.exception.TestConfigurationException;
import org.alfresco.utility.model.FileType;
import org.apache.chemistry.opencmis.client.api.SecondaryType;
import org.apache.poi.xslf.usermodel.XMLSlideShow;
import org.apache.poi.xslf.usermodel.XSLFShape;
import org.apache.poi.xslf.usermodel.XSLFTextShape;
import org.apache.poi.xssf.usermodel.XSSFCell;
import org.apache.poi.xssf.usermodel.XSSFRow;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.apache.poi.xwpf.usermodel.XWPFDocument;
import org.apache.poi.xwpf.usermodel.XWPFParagraph;
import org.springframework.beans.factory.annotation.Autowired;
import org.testng.Assert;

/**
 * DSL with all assertion available for {@link AOSWrapper}
 */

public class AOSAssertion extends DSLAssertion<AOSWrapper>
{
    @Autowired
    DataContent content;

    public AOSAssertion(AOSWrapper aosWrapper)
    {
        super(aosWrapper);
    }

    public AOSWrapper aosWrapper()
    {
        return getProtocol();
    }

    public AOSWrapper fileExistsInAOS() throws TestConfigurationException
    {
        String contentName = new File(aosWrapper().getAosLocation()).getPath();
        STEP(String.format("Verify that content '%s' exists in AOS", contentName));
        Assert.assertTrue(aosWrapper().withAosUtil().contentExistsInAos(contentName), String.format("File %s does not exist in AOS", contentName));

        return aosWrapper();
    }

    public AOSWrapper fileDoesNotExistInAOS() throws TestConfigurationException
    {
        String contentName = new File(aosWrapper().getAosLocation()).getPath();
        STEP(String.format("Verify that content '%s' exists in AOS", contentName));
        Assert.assertFalse(aosWrapper().withAosUtil().contentExistsInAos(contentName), String.format("File %s does not exist in AOS", contentName));

        return aosWrapper();
    }

    public AOSWrapper contentIs(String expectedContent) throws IOException, TestConfigurationException
    {
        STEP(String.format("AOS: Assert current content is '%s'", expectedContent));

        String actualContent = aosWrapper().getContent(aosWrapper().getAosLocation());

        Assert.assertEquals(actualContent, expectedContent, String.format("Content for file %s is not the expected one.", getProtocol().getLastResource()));

        return aosWrapper();
    }

    public AOSWrapper documentIsLocked()
    {
        STEP("AOS: Assert that document is locked for editing");
        List<SecondaryType> secondaryTypes = aosWrapper().withAosUtil().getCMISDocument().getSecondaryTypes();
        List<String> flags = new ArrayList<String>();

        for (int i = 0; i < secondaryTypes.size(); i++)
        {
            flags.add(secondaryTypes.get(i).getQueryName());
        }

        Assert.assertTrue(flags.contains("cm:lockable"), "Document is not locked, missing secondary type flag 'lockable'");

        return aosWrapper();
    }

    public AOSWrapper newPageExistsAndContentIsValid() throws IOException, TestConfigurationException {
        String resourcePath = aosWrapper().getAosLocation();
        FileInputStream file = new FileInputStream(new File(resourcePath));

        if (FileType.fromName(resourcePath).equals(FileType.MSWORD) || FileType.fromName(resourcePath).equals(FileType.MSWORD2007))
        {
            XWPFDocument document = new XWPFDocument(file);

            XWPFParagraph paragraph = document.getLastParagraph();
            Assert.assertTrue(paragraph.isPageBreak(), "No page break detected");
            document.close();
        }

        if (FileType.fromName(resourcePath).equals(FileType.MSEXCEL) || FileType.fromName(resourcePath).equals(FileType.MSEXCEL2007))
        {
            XSSFWorkbook wb = new XSSFWorkbook(file);
            XSSFRow row1 = wb.getSheetAt(1).getRow(1);
            XSSFCell cell1 = row1.getCell(1);

            Assert.assertEquals(wb.getNumberOfSheets(), 2, "Number of sheets is not correct");
            Assert.assertEquals(wb.getSheetAt(1).getSheetName(), "newSheet", "Sheet name is not correct");
            Assert.assertEquals(cell1.getStringCellValue(), aosWrapper().getDefaultFileContent(), "Sheet content is not correct");

            wb.close();
        }

        if (FileType.fromName(resourcePath).equals(FileType.MSPOWERPOINT) || FileType.fromName(resourcePath).equals(FileType.MSPOWERPOINT2007))
        {
            XMLSlideShow powerPoint = new XMLSlideShow(file);
            XSLFShape shape = powerPoint.getSlides().get(1).getShapes().get(0);

            if (shape instanceof XSLFTextShape)
            {
                XSLFTextShape textShape = (XSLFTextShape) shape;
                String text = textShape.getText();
                Assert.assertEquals(text, aosWrapper().getDefaultFileContent(), "Slide content is not correct");
            }

            Assert.assertEquals(powerPoint.getSlides().size(), 2, "Number of slides is not correct");

            powerPoint.close();
        }

        return aosWrapper();
    }


    /**
     * Verify the status for a specific action
     *
     * @param status
     * @return
     */
    public AOSWrapper hasStatus(int status)
    {
        Assert.assertEquals(aosWrapper().withAosUtil().getStatus(), status, String.format("Verify status"));
        return aosWrapper();
    }

}