/*
 * Copyright 2005-2015 Alfresco Software, Ltd.  All rights reserved.
 *
 * This file is part of a proprietary Alfresco module.
 *
 * License rights for this program are granted under the terms of the "Alfresco
 * Component License", which defines the permitted uses of the module.
 * License terms can be found in the file license.txt distributed with this module.
 */
package org.alfresco.officeservices.testclient.util;

import java.io.UnsupportedEncodingException;

/**
 * This class is similar to the well known <code>java.net.URLDecoder</code>, but it does decode
 * the slightly different URL path instead of the <code>application/x-www-form-urlencoded</code>
 * MIME format.<br/>
 * The single difference is that the plus sign <code>"+"</code> is <b>not</b> converted into the space character.
 * 
 * @author Stefan Kopf, xaldon Technologies GmbH.
 * 
 * @since 2.7
 */
public class URLPathDecoder
{

    /**
     * Private constructor to avoid instantiation.
     */
    private URLPathDecoder()
    {
        // just to prevent instantiation
    }
    
    /**
     * Decodes any consecutive sequences of the form <code>"%xy"</code>. The difference between this decoding and
     * the default <code>URLDecoder</code> class is that the plus sign (<code>+</code>) is <i>NOT</i> decoded.<br>
     * Some Office&trade; requests use this encoding scheme instead of <code>x-www-form-urlencoded</code>.
     * 
     * @param s the String to decode
     * @param encoding the character encoding to be used 
     * 
     * @return the newly decoded String
     * @throws UnsupportedEncodingException 
     */
    public static String decode(String s, String encoding) throws UnsupportedEncodingException
    {
        if(s == null)
        {
            return null;
        }
        StringBuilder result = new StringBuilder(s.length());
        // we need to decode the entire sequence of %-encoded bytes at once to be able to apply the encoding
        byte[] encodedBytesSequence = null;
        int ebsPos = 0;
        for(int i = 0; i < s.length(); i++)
        {
            char c = s.charAt(i);
            if((c == '%') && ((i+2) < s.length()))
            {
                if(encodedBytesSequence == null)
                {
                    encodedBytesSequence = new byte[s.length() / 3];
                }
                char c1 = s.charAt(i+1);
                char c2 = s.charAt(i+2);
                if(isHexDigit(c1) && isHexDigit(c2))
                {
                    encodedBytesSequence[ebsPos++] = (byte)decodeHexChar(c1, c2);
                    i += 2;
                }
                else
                {
                    if(ebsPos > 0)
                    {
                        result.append(new String(encodedBytesSequence, 0, ebsPos, encoding));
                        ebsPos = 0;
                    }
                    result.append(c);
                }
            }
            else
            {
                if(ebsPos > 0)
                {
                    result.append(new String(encodedBytesSequence, 0, ebsPos, encoding));
                    ebsPos = 0;
                }
                result.append(c);
            }
        }
        if(ebsPos > 0)
        {
            result.append(new String(encodedBytesSequence, 0, ebsPos, encoding));
        }
        return result.toString();
    }

    /**
     * Returns <code>true</code> if and only if the given byte represents a
     * valid hex digit (0..9 and (A..F or a..f) ).
     * 
     * @param test the input char to test
     * 
     * @return <code>true</code> if and only if the given byte represents a
     *     valid hex digit
     */
    private static boolean isHexDigit(char test)
    {
        return ((test >= '0' && test <= '9') || (test >= 'A' && test <= 'F') || (test >= 'a' && test <= 'f'));
    }

    /**
     * Decodes a char from two given hex digits.
     * 
     * @param c1 the most significant  hex digit
     * @param c2 the least significant lower hex digit
     * 
     * @return the character represented by the two given hex digits
     */
    private static char decodeHexChar(char c1, char c2)
    {
        if(c1 >= 'a' && c1 <= 'f')
        {
            c1 = (char) ('A' + (c1 - 'a'));
        }
        int result = (c1 >= 'A') ? ((c1 - 'A') + 10) : (c1 - '0');
        result = result * 16;
        if(c2 >= 'a' && c2 <= 'f')
        {
            c2 = (char) ('A' + (c2 - 'a'));
        }
        result += (c2 >= 'A') ? ((c2 - 'A') + 10) : (c2 - '0');
        return ((char) result);
    }
    
}
